import json
import os
import pickle

import numpy as np

# import scipy.io as sio
from sklearn import preprocessing

# from configurations import action_all, interested_raw_features, interested_compute_features
# from soccer_trace_len import process_trace
from generic.data_util import read_data, judge_home_away, ICEHOCKEY_GAME_FEATURES, \
    MANPOWER_FEATURE, OUTCOME_FEATURE, HA_FEATURE, ICEHOCKEY_ACTIONS, SOCCER_GAME_FEATURES, SOCCER_ACTIONS


class DataPreprocess:
    def __init__(self, source_data_dir, saved_data_dir, home_away_ids_dir=None, sports='ice-hockey'):
        self.source_data_dir = source_data_dir
        self.saved_data_dir = saved_data_dir
        self.sports = sports
        if home_away_ids_dir is not None:
            with open(home_away_ids_dir, 'r') as f:
                self.home_away_game_ids = json.load(f)
        else:
            self.home_away_game_ids = None
        assert os.path.exists(self.source_data_dir)

    def run_extract_name_date(self, save_dir='../soccer-data/game_dates_2017_2018.json'):
        from datetime import datetime as dt
        game_dates_all = []
        for file_name in os.listdir(self.source_data_dir):
            with open(os.path.join(self.source_data_dir, file_name)) as f:
                data = json.load(f)
            # tmp = data['gameDate'].split(" ")
            date_info = {"gameid": data['gameId'], "date": data['gameDate'].split(" ")[0],
                         "team1id": data['homeTeamId'], "team2id": data['awayTeamId']}
            game_dates_all.append(json.dumps(date_info))
        with open(save_dir, 'w') as f:
            f.write('[')
            for date_info in game_dates_all[:-1]:
                f.write(date_info + ',\n')
            f.write(game_dates_all[-1] + ']')

    def run_extract_event_data(self):
        features_all_games = {}
        rewards_all_games = {}
        _features_scale = []
        # for file_name in os.listdir(self.source_data_dir):
        for file_name in os.listdir(self.source_data_dir):
            print('Handling game {0}.'.format(file_name))
            events, game_id = read_data(source_data_dir=self.source_data_dir, file_name=file_name)
            if len(events) == 0:
                continue
            rewards_game, state_action_game = self.process_events(events=events, game_id=game_id)
            features_all_games.update({game_id: state_action_game})
            rewards_all_games.update({game_id: rewards_game})
            _features_scale += state_action_game

        feature_data_collect = np.asarray(_features_scale)
        feature_data_collect_max = np.max(feature_data_collect, axis=0)
        feature_data_collect_min = np.min(feature_data_collect, axis=0)
        scaler = preprocessing.StandardScaler().fit(np.asarray(feature_data_collect))

        feature_mean_dict = {}
        feature_std_dict = {}
        feature_max_dict = {}
        feature_min_dict = {}
        if self.sports == 'ice-hockey':
            feature_names = ICEHOCKEY_GAME_FEATURES + ICEHOCKEY_ACTIONS
        elif self.sports == 'soccer':
            feature_names = SOCCER_GAME_FEATURES + SOCCER_ACTIONS
        else:
            raise ValueError("Unknown sports: {0}".format(self.sports))
        for feature_idx in range(len(feature_names)):
            feature_name = feature_names[feature_idx]
            feature_mean = scaler.mean_[feature_idx]
            feature_std = scaler.scale_[feature_idx]
            feature_mean_dict.update({feature_name: feature_mean})
            feature_std_dict.update({feature_name: feature_std})
            feature_max_dict.update({feature_name: feature_data_collect_max[feature_idx]})
            feature_min_dict.update({feature_name: feature_data_collect_min[feature_idx]})

        if self.sports == 'ice-hockey':
            sport_dir = '../icehockey-data'
        elif self.sports == 'soccer':
            sport_dir = '../soccer-data'
        else:
            raise ValueError("Unknown sports: {0}".format(self.sports))
        with open(os.path.join(sport_dir, "feature_mean.json"), "w") as f:
            json.dump(feature_mean_dict, f)
        with open(os.path.join(sport_dir, "feature_std.json"), "w") as f:
            json.dump(feature_std_dict, f)
        with open(os.path.join(sport_dir, "feature_max.json"), "w") as f:
            json.dump(feature_max_dict, f)
        with open(os.path.join(sport_dir, "feature_min.json"), "w") as f:
            json.dump(feature_min_dict, f)
        for game_id in features_all_games.keys():
            state_action_game = features_all_games[game_id]
            state_action_game_standard = scaler.transform(np.asarray(state_action_game))
            if not os.path.exists(self.saved_data_dir + '/{0}'.format(game_id)):
                os.mkdir(self.saved_data_dir + '/{0}'.format(game_id))
            with open(self.saved_data_dir + '/{0}/state-action-data.pkl'.format(game_id), 'wb') as f:
                pickle.dump(state_action_game_standard, file=f)
            with open(self.saved_data_dir + '/{0}/reward-data.pkl'.format(game_id), 'wb') as f:
                pickle.dump(rewards_all_games[game_id], file=f)
        return scaler

    def process_events(self, events, game_id):  # each json contains many events
        rewards_game = []
        state_action_game = []
        previous_time = 0
        if self.sports == 'ice-hockey':
            home_away_pre = judge_home_away(home_away_game_ids=self.home_away_game_ids,
                                            teamId=events[0]['teamId'],
                                            gameId=game_id)
        elif self.sports == 'soccer':
            if events[0]['home_away'] == 'H':
                home_away_pre = 'home'
            elif events[0]['home_away'] == 'A':
                home_away_pre = 'away'
            else:
                raise ValueError("Unknown home_away.")
        else:
            raise ValueError("Unknown sports: {0}".format(self.sports))

        for index in range(0, len(events)):
            event = events[index]
            if self.sports == 'ice-hockey':
                action = event.get('name')
            elif self.sports == 'soccer':
                action = event.get('action')
            else:
                raise ValueError("Unknown sports: {0}".format(self.sports))
            one_hot_action = self.build_one_hot_action(action)

            if self.sports == 'ice-hockey':
                home_away = judge_home_away(home_away_game_ids=self.home_away_game_ids,
                                            teamId=events[index]['teamId'],
                                            gameId=game_id)
            elif self.sports == 'soccer':
                if events[index]['home_away'] == 'H':
                    home_away = 'home'
                elif events[index]['home_away'] == 'A':
                    home_away = 'away'
                else:
                    raise ValueError("Unknown home_away.")
            else:
                raise ValueError("Unknown sports: {0}".format(self.sports))

            if home_away == home_away_pre:
                switch_possession = False
            else:
                switch_possession = True

            home_away_pre = home_away

            if action == 'goal' and home_away == 'home':
                reward_event = [1, 0, 0]
            elif action == 'goal' and home_away == 'away':
                reward_event = [0, 1, 0]
            elif index == len(events) - 1:
                reward_event = [0, 0, 1]
            else:
                reward_event = [0, 0, 0]
            rewards_game.append(reward_event)
            feature_value_all = []
            if self.sports == 'ice-hockey':
                for feature_name in ICEHOCKEY_GAME_FEATURES:
                    if feature_name == 'xAdjCoord':
                        feature_value_all.append(event['xAdjCoord'])
                    elif feature_name == 'yAdjCoord':
                        feature_value_all.append(event['yAdjCoord'])
                    elif feature_name == 'scoreDifferential':
                        feature_value_all.append(event['scoreDifferential'])
                    elif feature_name == 'Penalty':
                        manpow_diff = event['manpowerSituation']
                        feature_value_all.append(MANPOWER_FEATURE[manpow_diff])
                    elif feature_name == 'duration':
                        duration = event['gameTime'] - previous_time
                        feature_value_all.append(duration)
                        previous_time = event['gameTime']
                    elif feature_name == 'velocity_x':
                        if duration == 0 or index == 0:
                            feature_value_all.append(0)
                        else:
                            coord_pre = events[index - 1]['xAdjCoord']
                            if switch_possession:  # change the coordinate system
                                coord_pre = - coord_pre
                            coord_now = events[index]['xAdjCoord']
                            velocity_x = (float(coord_now) - float(coord_pre)) / float(duration)
                            feature_value_all.append(velocity_x)
                    elif feature_name == 'velocity_y':
                        if duration == 0 or index == 0:
                            feature_value_all.append(0)
                        else:
                            coord_pre = events[index - 1]['yAdjCoord']
                            if switch_possession:  # change the coordinate system
                                coord_pre = - coord_pre
                            coord_now = events[index]['yAdjCoord']
                            velocity_y = (float(coord_now) - float(coord_pre)) / float(duration)
                            feature_value_all.append(velocity_y)
                    elif feature_name == 'time_remained':
                        feature_value_all.append(3600 - event['gameTime'])
                    elif feature_name == 'event_outcome':
                        feature_value_all.append(OUTCOME_FEATURE[event['outcome']])
                    elif feature_name == 'home':
                        feature_value_all.append(HA_FEATURE[home_away])
                    elif feature_name == 'away':
                        feature_value_all.append(1 - HA_FEATURE[home_away])
                    elif feature_name == 'angel2gate':
                        coord_x = events[index]['xAdjCoord']
                        coord_y = events[index]['yAdjCoord']
                        tanh = abs(coord_y) / (89 - coord_x)
                        feature_value_all.append(tanh)
                    else:
                        raise ValueError('Unknown feature {0}'.format(feature_name))
            elif self.sports == 'soccer':
                if index == 0:
                    duration = events[index].get('duration')
                else:
                    duration_pre = events[index - 1].get('duration')
                    duration_now = events[index].get('duration')
                    lt = events[index].get('lt')
                    if lt == 1:
                        duration = duration_now
                    else:
                        duration = duration_now - duration_pre
                for feature_name in SOCCER_GAME_FEATURES:
                    # if feature_name == 'home_away':
                    #     one_hot_vector = [0] * 2
                    #     if home_away == 'home':
                    #         one_hot_vector[0] = 1
                    #     elif home_away == 'away':
                    #         one_hot_vector[1] = 1
                    #     else:
                    #         raise ValueError("unclear home or away team")
                    #     feature_value_all += one_hot_vector
                    if feature_name == 'home':
                        feature_value_all.append(HA_FEATURE[home_away])
                    elif feature_name == 'away':
                        feature_value_all.append(1 - HA_FEATURE[home_away])
                    elif feature_name == 'velocity_x':
                        if index == 0 or duration == 0:
                            feature_value_all.append(float(0))
                        else:
                            coord_pre = events[index - 1].get('x')
                            if switch_possession:  # change the coordinate system
                                coord_pre = 100 - coord_pre
                            coord_now = events[index].get('x')
                            velocity_x = (float(coord_now) - float(coord_pre)) / float(duration)
                            feature_value_all.append(velocity_x)
                    elif feature_name == 'velocity_y':
                        if index == 0 or duration == 0:
                            feature_value_all.append(float(0))
                        else:
                            coord_pre = events[index - 1].get('y')
                            if switch_possession:  # change the coordinate system
                                coord_pre = 100 - coord_pre
                            coord_now = events[index].get('y')
                            velocity_y = (float(coord_now) - float(coord_pre)) / float(duration)
                            feature_value_all.append(velocity_y)
                    elif feature_name == 'distance_x':
                        coord_x = events[index].get('x')
                        distance_x = abs(100 - coord_x)
                        feature_value_all.append(distance_x)
                    elif feature_name == 'distance_y':
                        coord_y = events[index].get('y')
                        distance_y = abs(50 - coord_y)
                        feature_value_all.append(distance_y)
                    elif feature_name == 'distance_to_goal':
                        coord_x = events[index].get('x')
                        distance_x = abs(100 - coord_x)
                        coord_y = events[index].get('y')
                        distance_y = abs(50 - coord_y)
                        distance_to_goal = (distance_x ** 2 + distance_y ** 2) ** 0.5
                        feature_value_all.append(distance_to_goal)
                    elif feature_name == 'manPower':
                        man_power = event.get('manPower')
                        # if home_away == 'away':
                        #     manPower = -manPower  # change the agent
                        feature_value_all.append(man_power)
                    elif feature_name == 'scoreDiff':
                        score_diff = event.get('scoreDiff')
                        # if home_away == 'away':
                        #     scoreDiff = -scoreDiff  # change the agent
                        feature_value_all.append(score_diff)
                    elif feature_name in event.keys():
                        feature_value = event.get(feature_name)
                        feature_value = float(feature_value)  # for debug
                        feature_value_all.append(feature_value)
                    else:
                        raise ValueError('Unknown feature {0}'.format(feature_name))

            else:
                raise ValueError("Unknown sports: {0}".format(self.sports))
            state_action_game.append(np.asarray(feature_value_all + one_hot_action))
            # home_away[2] + left interested_raw_features[11] + one_hot_action[42] + velocity_x[1] + velocity_y[1] = [57]

        return rewards_game, state_action_game

    def build_one_hot_action(self, action):
        if self.sports == 'ice-hockey':
            one_hot_vector = [0] * 33
            index_action = ICEHOCKEY_ACTIONS.index(action)
        elif self.sports == 'soccer':
            one_hot_vector = [0] * 43
            index_action = SOCCER_ACTIONS.index(action)
        else:
            raise ValueError("Unknown sports: {0}".format(self.sports))
        one_hot_vector[index_action] = 1
        return one_hot_vector

    def run_extract_game_info(self, save_dir):
        game_info_dict = {}
        for file_name in os.listdir(self.source_data_dir):
            with open(os.path.join(self.source_data_dir, file_name)) as f:
                data = json.load(f)
            game_info = {"team1id": data['homeTeamId'], "team2id": data['awayTeamId']}
            game_info_dict.update({data['gameId']: game_info})
        with open(save_dir, 'w') as f:
            json.dump(game_info_dict, fp=f)
